#!/usr/bin/env bash
#===============================================================================
# bootstrap.sh — Wadsworth AI PC One-Line Installer
#
# Downloads and runs the Wadsworth installer from a remote server.
# Designed to be curl'd and piped to bash.
#
# Usage:
#   curl -sSL https://your-server.com/wadsworth/bootstrap.sh | sudo bash
#
# Or with custom URL:
#   curl -sSL https://your-server.com/wadsworth/bootstrap.sh | sudo WADSWORTH_URL=https://custom.com bash
#
# Environment Variables:
#   WADSWORTH_URL  - Base URL for downloads (default: must be set or edit RELEASE_URL below)
#   WADSWORTH_SKIP_CONFIRM - Set to 1 to skip confirmation prompt
#
#===============================================================================

set -euo pipefail

#-------------------------------------------------------------------------------
# Configuration — EDIT THIS FOR YOUR SERVER
#-------------------------------------------------------------------------------
# Default release URL - change this to your VPS URL
DEFAULT_RELEASE_URL="https://your-vps.example.com/wadsworth/wadsworth-release.tar.gz"

# Use environment variable if set, otherwise use default
RELEASE_URL="${WADSWORTH_URL:-${DEFAULT_RELEASE_URL}}"

#-------------------------------------------------------------------------------
# Colors
#-------------------------------------------------------------------------------
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
BOLD='\033[1m'
NC='\033[0m'

#-------------------------------------------------------------------------------
# Functions
#-------------------------------------------------------------------------------
error() {
    echo -e "${RED}ERROR:${NC} $1" >&2
    exit 1
}

info() {
    echo -e "${BLUE}==>${NC} $1"
}

success() {
    echo -e "${GREEN}==>${NC} $1"
}

warn() {
    echo -e "${YELLOW}WARNING:${NC} $1"
}

#-------------------------------------------------------------------------------
# Pre-flight Checks
#-------------------------------------------------------------------------------
check_root() {
    if [[ "${EUID}" -ne 0 ]]; then
        error "This script must be run as root. Use: curl ... | sudo bash"
    fi
}

check_os() {
    if [[ "$(uname -s)" != "Linux" ]]; then
        error "This script only runs on Linux"
    fi

    if [[ -f /etc/os-release ]]; then
        source /etc/os-release
        case "${ID}" in
            linuxmint|ubuntu|debian)
                info "Detected: ${PRETTY_NAME}"
                ;;
            *)
                warn "Untested distribution: ${ID}. Proceeding anyway..."
                ;;
        esac
    fi
}

check_commands() {
    local missing=()
    for cmd in curl tar; do
        if ! command -v "${cmd}" &>/dev/null; then
            missing+=("${cmd}")
        fi
    done

    if [[ ${#missing[@]} -gt 0 ]]; then
        error "Missing required commands: ${missing[*]}"
    fi
}

#-------------------------------------------------------------------------------
# Main Installation
#-------------------------------------------------------------------------------
main() {
    echo ""
    echo -e "${BOLD}========================================${NC}"
    echo -e "${BOLD}  Wadsworth AI PC Installer${NC}"
    echo -e "${BOLD}========================================${NC}"
    echo ""

    check_root
    check_os
    check_commands

    info "Release URL: ${RELEASE_URL}"
    echo ""

    # Check if URL is the placeholder
    if [[ "${RELEASE_URL}" == *"your-vps.example.com"* ]]; then
        error "Release URL not configured. Set WADSWORTH_URL environment variable or edit bootstrap.sh"
    fi

    # Confirmation
    if [[ "${WADSWORTH_SKIP_CONFIRM:-0}" != "1" ]]; then
        echo "This will:"
        echo "  1. Download the Wadsworth release package"
        echo "  2. Install required packages (firefox, vlc, libreoffice, etc.)"
        echo "  3. Copy Wadsworth payload to /opt/wadsworth_payload"
        echo "  4. Set up the firstboot service"
        echo ""
        read -p "Continue? [y/N] " -n 1 -r
        echo ""
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            echo "Aborted."
            exit 0
        fi
    fi

    # Create temp directory
    WORK_DIR=$(mktemp -d)
    trap "rm -rf ${WORK_DIR}" EXIT

    # Download release
    info "Downloading release..."
    if ! curl -sSL -o "${WORK_DIR}/release.tar.gz" "${RELEASE_URL}"; then
        error "Failed to download release from ${RELEASE_URL}"
    fi

    # Verify download
    if [[ ! -s "${WORK_DIR}/release.tar.gz" ]]; then
        error "Downloaded file is empty"
    fi

    info "Download complete: $(du -h "${WORK_DIR}/release.tar.gz" | cut -f1)"

    # Extract
    info "Extracting..."
    cd "${WORK_DIR}"
    if ! tar -xzf release.tar.gz; then
        error "Failed to extract release"
    fi

    # Find extracted directory
    RELEASE_DIR=$(find . -maxdepth 1 -type d -name "wadsworth*" | head -1)
    if [[ -z "${RELEASE_DIR}" ]]; then
        error "Could not find extracted release directory"
    fi

    # Run installer
    info "Running installer..."
    echo ""

    INSTALL_SCRIPT="${RELEASE_DIR}/src/install.sh"
    if [[ ! -f "${INSTALL_SCRIPT}" ]]; then
        error "install.sh not found in release"
    fi

    chmod +x "${INSTALL_SCRIPT}"

    # Run install.sh from its directory so relative paths work
    cd "${RELEASE_DIR}/src"
    if bash install.sh --verbose; then
        echo ""
        success "Installation complete!"
        echo ""
        echo "Next steps:"
        echo "  1. Reboot the machine (firstboot will run automatically)"
        echo "  2. Or run firstboot manually: sudo /usr/local/sbin/wadsworth-firstboot.sh"
        echo ""
    else
        error "Installation failed"
    fi
}

main "$@"
