#!/usr/bin/env bash
#===============================================================================
# Wadsworth AI PCs — Quality Control Script (Production v1.0)
#
# This script runs on a fully deployed unit (after firstboot has completed)
# to verify it's ready to ship. It's the final gate before a unit goes to
# a customer.
#
# Usage:
#   sudo ./run_qc.sh [OPTIONS]
#
# Options:
#   --verbose   Extra detail in output
#   --json      Output results as JSON (for automation)
#   --help      Show this help message
#
# Exit codes:
#   0 = PASS (all checks passed)
#   1 = FAIL (one or more checks failed)
#   2 = ERROR (script couldn't run properly)
#===============================================================================

set -euo pipefail

#-------------------------------------------------------------------------------
# Configuration
#-------------------------------------------------------------------------------
readonly SCRIPT_VERSION="1.0"
readonly LOG_DIR="/var/log/wadsworth"
readonly LOG_FILE="${LOG_DIR}/qc_run.log"
readonly MARKER_FILE="/var/lib/wadsworth/firstboot.done"
readonly PAYLOAD_DIR="/opt/wadsworth"

#-------------------------------------------------------------------------------
# Runtime flags
#-------------------------------------------------------------------------------
VERBOSE=false
JSON_OUTPUT=false

#-------------------------------------------------------------------------------
# Results tracking
#-------------------------------------------------------------------------------
declare -a CHECK_NAMES=()
declare -a CHECK_RESULTS=()
declare -a CHECK_DETAILS=()

PASS_COUNT=0
FAIL_COUNT=0
WARN_COUNT=0

# Hardware info
HW_MODEL=""
HW_SERIAL=""
HW_CPU=""
HW_RAM=""
HW_DISK_SIZE=""
HW_DISK_USED=""
HW_WIFI=""
HW_AUDIO=""
HW_HOSTNAME=""

#-------------------------------------------------------------------------------
# Logging Functions
#-------------------------------------------------------------------------------
log() {
    local level="${1:-INFO}"
    shift
    local message="$*"
    local timestamp
    timestamp="$(date -Iseconds)"

    # Log to file
    if [[ -d "${LOG_DIR}" ]]; then
        echo "[${timestamp}] [${level}] ${message}" >> "${LOG_FILE}"
    fi
}

log_verbose() {
    if [[ "${VERBOSE}" == true ]] && [[ "${JSON_OUTPUT}" == false ]]; then
        echo "  [DETAIL] $*"
    fi
}

#-------------------------------------------------------------------------------
# Output Functions
#-------------------------------------------------------------------------------
print_header() {
    if [[ "${JSON_OUTPUT}" == false ]]; then
        echo ""
        echo "=========================================="
        echo "  Wadsworth AI PCs — Quality Control"
        echo "  Version ${SCRIPT_VERSION}"
        echo "=========================================="
        echo ""
    fi
}

print_section() {
    if [[ "${JSON_OUTPUT}" == false ]]; then
        echo ""
        echo "--- $1 ---"
    fi
}

print_result() {
    local name="$1"
    local result="$2"
    local detail="${3:-}"

    # Store for JSON output
    CHECK_NAMES+=("${name}")
    CHECK_RESULTS+=("${result}")
    CHECK_DETAILS+=("${detail}")

    # Update counts
    case "${result}" in
        PASS) ((PASS_COUNT++)) || true ;;
        FAIL) ((FAIL_COUNT++)) || true ;;
        WARN) ((WARN_COUNT++)) || true ;;
    esac

    # Console output
    if [[ "${JSON_OUTPUT}" == false ]]; then
        local color=""
        local reset="\033[0m"
        case "${result}" in
            PASS) color="\033[32m" ;;  # Green
            FAIL) color="\033[31m" ;;  # Red
            WARN) color="\033[33m" ;;  # Yellow
        esac
        printf "  [${color}%s${reset}] %s" "${result}" "${name}"
        if [[ -n "${detail}" ]] && [[ "${VERBOSE}" == true ]]; then
            printf " (%s)" "${detail}"
        fi
        printf "\n"
    fi

    # Log result
    log "INFO" "QC Check: ${name} = ${result} ${detail}"
}

print_hw_info() {
    if [[ "${JSON_OUTPUT}" == false ]]; then
        echo "  Model:    ${HW_MODEL}"
        echo "  Serial:   ${HW_SERIAL}"
        echo "  Hostname: ${HW_HOSTNAME}"
        echo "  CPU:      ${HW_CPU}"
        echo "  RAM:      ${HW_RAM}"
        echo "  Disk:     ${HW_DISK_SIZE} (${HW_DISK_USED} used)"
        echo "  Wi-Fi:    ${HW_WIFI}"
        echo "  Audio:    ${HW_AUDIO}"
    fi
}

#-------------------------------------------------------------------------------
# JSON Output
#-------------------------------------------------------------------------------
output_json() {
    local final_result="$1"
    local timestamp
    timestamp="$(date -Iseconds)"

    echo "{"
    echo "  \"version\": \"${SCRIPT_VERSION}\","
    echo "  \"timestamp\": \"${timestamp}\","
    echo "  \"verdict\": \"${final_result}\","
    echo "  \"summary\": {"
    echo "    \"pass\": ${PASS_COUNT},"
    echo "    \"fail\": ${FAIL_COUNT},"
    echo "    \"warn\": ${WARN_COUNT}"
    echo "  },"
    echo "  \"hardware\": {"
    echo "    \"model\": \"${HW_MODEL}\","
    echo "    \"serial\": \"${HW_SERIAL}\","
    echo "    \"hostname\": \"${HW_HOSTNAME}\","
    echo "    \"cpu\": \"${HW_CPU}\","
    echo "    \"ram\": \"${HW_RAM}\","
    echo "    \"disk_size\": \"${HW_DISK_SIZE}\","
    echo "    \"disk_used\": \"${HW_DISK_USED}\","
    echo "    \"wifi\": \"${HW_WIFI}\","
    echo "    \"audio\": \"${HW_AUDIO}\""
    echo "  },"
    echo "  \"checks\": ["

    local count=${#CHECK_NAMES[@]}
    for ((i=0; i<count; i++)); do
        local comma=","
        if [[ $((i + 1)) -eq ${count} ]]; then
            comma=""
        fi
        # Escape quotes in detail
        local escaped_detail="${CHECK_DETAILS[$i]//\"/\\\"}"
        echo "    {"
        echo "      \"name\": \"${CHECK_NAMES[$i]}\","
        echo "      \"result\": \"${CHECK_RESULTS[$i]}\","
        echo "      \"detail\": \"${escaped_detail}\""
        echo "    }${comma}"
    done

    echo "  ]"
    echo "}"
}

#-------------------------------------------------------------------------------
# Usage/Help
#-------------------------------------------------------------------------------
show_help() {
    cat << 'EOF'
Wadsworth AI PCs — Quality Control Script

Usage:
    sudo ./run_qc.sh [OPTIONS]

Options:
    --verbose   Extra detail in output
    --json      Output results as JSON (for automation)
    --help      Show this help message

Description:
    This script verifies a deployed Wadsworth AI PC is ready for shipping.
    It checks:
    - Hardware info (CPU, RAM, disk, Wi-Fi, audio)
    - Firstboot completion
    - Payload installation (/opt/wadsworth/)
    - Desktop shortcuts
    - Required applications
    - System configuration

Exit Codes:
    0 = PASS (all checks passed)
    1 = FAIL (one or more checks failed)
    2 = ERROR (script couldn't run properly)

Examples:
    sudo ./run_qc.sh              # Normal QC run
    sudo ./run_qc.sh --verbose    # Detailed output
    sudo ./run_qc.sh --json       # JSON output for automation

EOF
    exit 0
}

#-------------------------------------------------------------------------------
# Argument Parsing
#-------------------------------------------------------------------------------
parse_args() {
    while [[ $# -gt 0 ]]; do
        case "$1" in
            --verbose)
                VERBOSE=true
                shift
                ;;
            --json)
                JSON_OUTPUT=true
                shift
                ;;
            --help|-h)
                show_help
                ;;
            *)
                echo "[ERROR] Unknown option: $1"
                echo "Use --help for usage information."
                exit 2
                ;;
        esac
    done
}

#-------------------------------------------------------------------------------
# Pre-flight Checks
#-------------------------------------------------------------------------------
check_root() {
    if [[ "${EUID}" -ne 0 ]]; then
        if [[ "${JSON_OUTPUT}" == true ]]; then
            echo "{\"error\": \"Must be run as root\", \"exit_code\": 2}"
        else
            echo "[ERROR] This script must be run as root"
            echo "Usage: sudo $0 [OPTIONS]"
        fi
        exit 2
    fi
}

init_logging() {
    mkdir -p "${LOG_DIR}" 2>/dev/null || true
    touch "${LOG_FILE}" 2>/dev/null || true
    log "INFO" "=========================================="
    log "INFO" "QC Script Started"
    log "INFO" "=========================================="
}

#-------------------------------------------------------------------------------
# Hardware Info Collection
#-------------------------------------------------------------------------------
collect_hardware_info() {
    if [[ "${JSON_OUTPUT}" == false ]]; then
        print_section "Hardware Information"
    fi

    # Model
    if [[ -f /sys/devices/virtual/dmi/id/product_name ]]; then
        HW_MODEL="$(cat /sys/devices/virtual/dmi/id/product_name 2>/dev/null || echo "Unknown")"
    else
        HW_MODEL="Unknown"
    fi

    # Serial
    if [[ -f /sys/devices/virtual/dmi/id/product_serial ]]; then
        HW_SERIAL="$(cat /sys/devices/virtual/dmi/id/product_serial 2>/dev/null || echo "Unknown")"
        # Hide common placeholder values
        if [[ "${HW_SERIAL}" == "To Be Filled By O.E.M." ]] || [[ "${HW_SERIAL}" == "None" ]]; then
            HW_SERIAL="Not available"
        fi
    else
        HW_SERIAL="Unknown"
    fi

    # Hostname
    HW_HOSTNAME="$(hostname 2>/dev/null || echo "Unknown")"

    # CPU
    if [[ -f /proc/cpuinfo ]]; then
        HW_CPU="$(grep "model name" /proc/cpuinfo 2>/dev/null | head -1 | cut -d: -f2 | xargs || echo "Unknown")"
    else
        HW_CPU="Unknown"
    fi

    # RAM
    if command -v free &>/dev/null; then
        HW_RAM="$(free -h 2>/dev/null | awk '/^Mem:/ {print $2}' || echo "Unknown")"
    else
        HW_RAM="Unknown"
    fi

    # Disk
    if command -v df &>/dev/null; then
        local disk_info
        disk_info="$(df -h / 2>/dev/null | awk 'NR==2 {print $2, $5}')"
        HW_DISK_SIZE="$(echo "${disk_info}" | awk '{print $1}')"
        HW_DISK_USED="$(echo "${disk_info}" | awk '{print $2}')"
    else
        HW_DISK_SIZE="Unknown"
        HW_DISK_USED="Unknown"
    fi

    # Wi-Fi adapter
    if [[ -d /sys/class/net ]]; then
        local wifi_adapters
        wifi_adapters="$(ls /sys/class/net/ 2>/dev/null | grep -E '^wl' || true)"
        if [[ -n "${wifi_adapters}" ]]; then
            HW_WIFI="Present (${wifi_adapters})"
        else
            HW_WIFI="Not detected"
        fi
    else
        HW_WIFI="Unknown"
    fi

    # Audio
    if command -v aplay &>/dev/null; then
        local audio_cards
        audio_cards="$(aplay -l 2>/dev/null | grep -c "^card" || echo "0")"
        if [[ "${audio_cards}" -gt 0 ]]; then
            HW_AUDIO="Present (${audio_cards} card(s))"
        else
            HW_AUDIO="No cards detected"
        fi
    else
        HW_AUDIO="aplay not available"
    fi

    print_hw_info
    log "INFO" "Hardware: Model=${HW_MODEL}, Serial=${HW_SERIAL}, CPU=${HW_CPU}, RAM=${HW_RAM}"
}

#-------------------------------------------------------------------------------
# Primary User Detection
#-------------------------------------------------------------------------------
get_primary_user() {
    local username uid home

    while IFS=: read -r username _ uid _ _ home _; do
        if [[ ${uid} -ge 1000 ]] && [[ -d "${home}" ]] && [[ "${username}" != "nobody" ]]; then
            echo "${username}"
            return 0
        fi
    done < /etc/passwd

    return 1
}

#-------------------------------------------------------------------------------
# Check Functions
#-------------------------------------------------------------------------------
check_firstboot() {
    print_section "Firstboot Verification"

    # Check marker file
    if [[ -f "${MARKER_FILE}" ]]; then
        local marker_date
        marker_date="$(cat "${MARKER_FILE}" 2>/dev/null || echo "unknown")"
        print_result "Firstboot completed" "PASS" "completed: ${marker_date}"
    else
        print_result "Firstboot completed" "FAIL" "marker file missing"
    fi

    # Check service is disabled (it should be after running)
    if command -v systemctl &>/dev/null; then
        if systemctl is-enabled wadsworth-firstboot.service &>/dev/null; then
            # Service is still enabled — that's expected, it's a oneshot
            # Check if it has run successfully
            if systemctl is-active wadsworth-firstboot.service &>/dev/null; then
                print_result "Firstboot service status" "PASS" "completed"
            else
                local service_status
                service_status="$(systemctl is-active wadsworth-firstboot.service 2>/dev/null || echo "unknown")"
                if [[ "${service_status}" == "inactive" ]] && [[ -f "${MARKER_FILE}" ]]; then
                    print_result "Firstboot service status" "PASS" "inactive (completed)"
                else
                    print_result "Firstboot service status" "WARN" "status: ${service_status}"
                fi
            fi
        else
            # Service disabled or not found
            if [[ -f "${MARKER_FILE}" ]]; then
                print_result "Firstboot service status" "PASS" "disabled (completed)"
            else
                print_result "Firstboot service status" "FAIL" "not enabled, marker missing"
            fi
        fi
    else
        print_result "Firstboot service status" "WARN" "systemctl not available"
    fi
}

check_payload() {
    print_section "Payload Verification"

    # Required files
    local required_files=(
        "ai_hub/index.html"
        "welcome_center/index.html"
        "prompt_pack/prompt_pack.html"
    )

    for file in "${required_files[@]}"; do
        local full_path="${PAYLOAD_DIR}/${file}"
        if [[ -f "${full_path}" ]]; then
            print_result "Payload: ${file}" "PASS" ""
            log_verbose "Found: ${full_path}"
        else
            print_result "Payload: ${file}" "FAIL" "not found"
        fi
    done

    # Required directories
    local required_dirs=(
        "desktop_shortcuts"
        "docs"
        "assets"
    )

    for dir in "${required_dirs[@]}"; do
        local full_path="${PAYLOAD_DIR}/${dir}"
        if [[ -d "${full_path}" ]]; then
            local file_count
            file_count="$(find "${full_path}" -type f 2>/dev/null | wc -l | xargs)"
            print_result "Payload: ${dir}/" "PASS" "${file_count} files"
        else
            print_result "Payload: ${dir}/" "FAIL" "directory not found"
        fi
    done
}

check_desktop_shortcuts() {
    print_section "Desktop Shortcuts"

    local primary_user primary_home
    if primary_user="$(get_primary_user)"; then
        primary_home="$(getent passwd "${primary_user}" | cut -d: -f6)"
        log_verbose "Primary user: ${primary_user}, home: ${primary_home}"
    else
        print_result "Primary user detection" "FAIL" "no user with UID >= 1000"
        return
    fi

    local desktop_dir="${primary_home}/Desktop"
    local ai_dir="${desktop_dir}/AI"

    # Check Desktop directory
    if [[ ! -d "${desktop_dir}" ]]; then
        print_result "Desktop directory" "FAIL" "not found"
        return
    fi

    # Check Welcome Center shortcut
    if [[ -f "${desktop_dir}/welcome-center.desktop" ]]; then
        print_result "Welcome Center shortcut" "PASS" ""
    else
        print_result "Welcome Center shortcut" "FAIL" "not found on Desktop"
    fi

    # Check AI folder
    if [[ -d "${ai_dir}" ]]; then
        print_result "AI folder on Desktop" "PASS" ""
    else
        print_result "AI folder on Desktop" "FAIL" "not found"
        return
    fi

    # Check AI shortcuts
    local ai_shortcuts=(
        "chatgpt.desktop"
        "claude.desktop"
        "gemini.desktop"
        "ai-hub.desktop"
    )

    for shortcut in "${ai_shortcuts[@]}"; do
        if [[ -f "${ai_dir}/${shortcut}" ]]; then
            print_result "AI shortcut: ${shortcut}" "PASS" ""
        else
            print_result "AI shortcut: ${shortcut}" "FAIL" "not found"
        fi
    done
}

check_applications() {
    print_section "Application Verification"

    # Required applications (FAIL if missing)
    local required_apps=(
        "firefox"
        "vlc"
        "libreoffice"
        "timeshift"
    )

    for app in "${required_apps[@]}"; do
        if command -v "${app}" &>/dev/null; then
            local version
            version="$(${app} --version 2>/dev/null | head -1 || echo "installed")"
            print_result "Application: ${app}" "PASS" "${version}"
        else
            print_result "Application: ${app}" "FAIL" "not found"
        fi
    done

    # Chromium (either name)
    if command -v chromium-browser &>/dev/null; then
        print_result "Application: chromium" "PASS" "chromium-browser"
    elif command -v chromium &>/dev/null; then
        print_result "Application: chromium" "PASS" "chromium"
    else
        print_result "Application: chromium" "FAIL" "not found"
    fi

    # RustDesk (WARN if missing, not FAIL)
    if command -v rustdesk &>/dev/null; then
        print_result "Application: rustdesk" "PASS" ""
    else
        print_result "Application: rustdesk" "WARN" "not installed (manual install required)"
    fi
}

check_system() {
    print_section "System Checks"

    # Hostname check
    local hostname
    hostname="$(hostname 2>/dev/null || echo "")"
    if [[ "${hostname}" == wadsworth-* ]]; then
        print_result "Hostname format" "PASS" "${hostname}"
    else
        print_result "Hostname format" "FAIL" "expected wadsworth-*, got: ${hostname}"
    fi

    # Machine-ID check
    if [[ -f /etc/machine-id ]]; then
        local machine_id
        machine_id="$(cat /etc/machine-id 2>/dev/null || echo "")"
        if [[ -n "${machine_id}" ]]; then
            local short_id="${machine_id:0:8}..."
            print_result "Machine-ID exists" "PASS" "${short_id}"
        else
            print_result "Machine-ID exists" "FAIL" "file empty"
        fi
    else
        print_result "Machine-ID exists" "FAIL" "file not found"
    fi

    # Network connectivity
    if command -v ping &>/dev/null; then
        if ping -c 1 -W 3 8.8.8.8 &>/dev/null; then
            print_result "Network connectivity" "PASS" "ping 8.8.8.8 OK"
        else
            print_result "Network connectivity" "FAIL" "ping 8.8.8.8 failed"
        fi
    else
        print_result "Network connectivity" "WARN" "ping not available"
    fi
}

#-------------------------------------------------------------------------------
# Final Summary
#-------------------------------------------------------------------------------
print_summary() {
    local final_result

    if [[ ${FAIL_COUNT} -gt 0 ]]; then
        final_result="FAIL"
    else
        final_result="PASS"
    fi

    if [[ "${JSON_OUTPUT}" == true ]]; then
        output_json "${final_result}"
    else
        echo ""
        echo "=========================================="
        echo "  QC Results Summary"
        echo "=========================================="
        echo ""
        printf "  PASS: %d\n" "${PASS_COUNT}"
        printf "  FAIL: %d\n" "${FAIL_COUNT}"
        printf "  WARN: %d\n" "${WARN_COUNT}"
        echo ""

        if [[ "${final_result}" == "PASS" ]]; then
            echo -e "  \033[32m*** VERDICT: PASS ***\033[0m"
            echo ""
            echo "  This unit is ready for shipping."
        else
            echo -e "  \033[31m*** VERDICT: FAIL ***\033[0m"
            echo ""
            echo "  This unit has ${FAIL_COUNT} failing check(s)."
            echo "  Review the failed items above before shipping."
        fi

        if [[ ${WARN_COUNT} -gt 0 ]]; then
            echo ""
            echo "  Note: ${WARN_COUNT} warning(s) detected (non-blocking)."
        fi

        echo ""
        echo "=========================================="
    fi

    log "INFO" "QC Complete: ${final_result} (PASS=${PASS_COUNT}, FAIL=${FAIL_COUNT}, WARN=${WARN_COUNT})"

    # Return appropriate exit code
    if [[ "${final_result}" == "PASS" ]]; then
        return 0
    else
        return 1
    fi
}

#-------------------------------------------------------------------------------
# Main Execution
#-------------------------------------------------------------------------------
main() {
    # Parse command-line arguments
    parse_args "$@"

    # Pre-flight
    check_root
    init_logging

    # Header
    print_header

    # Collect hardware info
    collect_hardware_info

    # Run all checks
    check_firstboot
    check_payload
    check_desktop_shortcuts
    check_applications
    check_system

    # Summary and exit
    print_summary
    exit $?
}

# Run main function
main "$@"
