#!/usr/bin/env bash
#===============================================================================
# Wadsworth AI PCs — Golden Image Installer (Production v1.0)
#
# This script runs on the golden image machine BEFORE capturing the image.
# It installs packages, copies the payload, and sets up the firstboot service.
#
# Usage:
#   sudo ./install.sh [OPTIONS]
#
# Options:
#   --dry-run   Show what would be done without making changes
#   --verbose   Extra logging output
#   --help      Show this help message
#
# IMPORTANT:
# - Must be run as root
# - Do NOT reboot after running — firstboot runs on deployed units only
# - RustDesk must be installed manually from .deb (not in apt repos)
#===============================================================================

set -euo pipefail

#-------------------------------------------------------------------------------
# Configuration
#-------------------------------------------------------------------------------
readonly SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly LOG_DIR="/var/log/wadsworth"
readonly LOG_FILE="${LOG_DIR}/install.log"
readonly PAYLOAD_SOURCE="${SCRIPT_DIR}"
readonly PAYLOAD_DEST="/opt/wadsworth_payload"
readonly FIRSTBOOT_SCRIPT_SRC="${SCRIPT_DIR}/firstboot/firstboot.sh"
readonly FIRSTBOOT_SCRIPT_DST="/usr/local/sbin/wadsworth-firstboot.sh"
readonly FIRSTBOOT_SERVICE_SRC="${SCRIPT_DIR}/firstboot/wadsworth-firstboot.service"
readonly FIRSTBOOT_SERVICE_DST="/etc/systemd/system/wadsworth-firstboot.service"

# Packages to install (RustDesk excluded — requires manual .deb install)
readonly PACKAGES=(
    "firefox"
    "vlc"
    "libreoffice"
    "timeshift"
    "deja-dup"
    "simple-scan"
    "system-config-printer"
)

# Commands to verify after installation (package:command pairs)
readonly PACKAGE_COMMANDS=(
    "firefox:firefox"
    "vlc:vlc"
    "libreoffice:libreoffice"
    "timeshift:timeshift"
    "deja-dup:deja-dup"
    "simple-scan:simple-scan"
    "system-config-printer:system-config-printer"
)

# Payload directories to copy
readonly PAYLOAD_DIRS=(
    "ai_hub"
    "prompt_pack"
    "welcome_center"
    "docs"
    "assets"
    "desktop_shortcuts"
    "bin"
)

#-------------------------------------------------------------------------------
# Runtime flags
#-------------------------------------------------------------------------------
DRY_RUN=false
VERBOSE=false

# Tracking for summary
PACKAGES_INSTALLED=()
FILES_DEPLOYED=()
WARNINGS=()

#-------------------------------------------------------------------------------
# Logging Functions
#-------------------------------------------------------------------------------
log() {
    local level="${1:-INFO}"
    shift
    local message="$*"
    local timestamp
    timestamp="$(date -Iseconds)"

    # Always log to file (if not dry-run and log dir exists)
    if [[ "${DRY_RUN}" == false ]] && [[ -d "${LOG_DIR}" ]]; then
        echo "[${timestamp}] [${level}] ${message}" >> "${LOG_FILE}"
    fi

    # Print to console
    echo "[${level}] ${message}"
}

log_verbose() {
    if [[ "${VERBOSE}" == true ]]; then
        log "DEBUG" "$@"
    fi
}

log_dry() {
    if [[ "${DRY_RUN}" == true ]]; then
        echo "[DRY-RUN] Would: $*"
    fi
}

#-------------------------------------------------------------------------------
# Usage/Help
#-------------------------------------------------------------------------------
show_help() {
    cat << 'EOF'
Wadsworth AI PCs — Golden Image Installer

Usage:
    sudo ./install.sh [OPTIONS]

Options:
    --dry-run   Show what would be done without making changes
    --verbose   Extra logging output
    --help      Show this help message

Description:
    This script prepares a golden image for Wadsworth AI PCs by:
    1. Installing required packages (Firefox, VLC, LibreOffice, etc.)
    2. Copying the Wadsworth payload to /opt/wadsworth_payload/
    3. Installing and enabling the firstboot service

    After running this script:
    - Do NOT reboot (firstboot runs on deployed units only)
    - Install RustDesk manually from .deb file
    - Capture the golden image with Clonezilla

Examples:
    sudo ./install.sh                    # Normal installation
    sudo ./install.sh --dry-run          # Preview without changes
    sudo ./install.sh --verbose          # Detailed output
    sudo ./install.sh --dry-run --verbose # Preview with details

EOF
    exit 0
}

#-------------------------------------------------------------------------------
# Argument Parsing
#-------------------------------------------------------------------------------
parse_args() {
    while [[ $# -gt 0 ]]; do
        case "$1" in
            --dry-run)
                DRY_RUN=true
                shift
                ;;
            --verbose)
                VERBOSE=true
                shift
                ;;
            --help|-h)
                show_help
                ;;
            *)
                log "ERROR" "Unknown option: $1"
                echo "Use --help for usage information."
                exit 1
                ;;
        esac
    done
}

#-------------------------------------------------------------------------------
# Pre-flight Checks
#-------------------------------------------------------------------------------
check_root() {
    if [[ "${EUID}" -ne 0 ]]; then
        log "ERROR" "This script must be run as root"
        echo "Usage: sudo $0 [OPTIONS]"
        exit 1
    fi
    log "INFO" "Running as root"
}

check_os() {
    log "INFO" "Checking operating system..."

    local os_name=""
    local os_version=""

    if [[ -f /etc/os-release ]]; then
        # shellcheck source=/dev/null
        source /etc/os-release
        os_name="${NAME:-unknown}"
        os_version="${VERSION_ID:-unknown}"
    fi

    log_verbose "Detected OS: ${os_name} ${os_version}"

    if [[ "${os_name}" != *"Linux Mint"* ]]; then
        log "WARN" "This script is designed for Linux Mint"
        log "WARN" "Detected: ${os_name} ${os_version}"
        log "WARN" "Continuing anyway, but some features may not work correctly"
        WARNINGS+=("Running on non-Mint OS: ${os_name}")
    else
        log "INFO" "Verified: ${os_name} ${os_version}"
    fi
}

init_logging() {
    if [[ "${DRY_RUN}" == true ]]; then
        log_dry "Create log directory ${LOG_DIR}"
        return 0
    fi

    mkdir -p "${LOG_DIR}"
    touch "${LOG_FILE}"
    chmod 644 "${LOG_FILE}"

    log "INFO" "=========================================="
    log "INFO" "Wadsworth Golden Image Installer Starting"
    log "INFO" "=========================================="
    log "INFO" "Script directory: ${SCRIPT_DIR}"
    log "INFO" "Dry-run mode: ${DRY_RUN}"
    log "INFO" "Verbose mode: ${VERBOSE}"
}

#-------------------------------------------------------------------------------
# Package Installation
#-------------------------------------------------------------------------------
install_packages() {
    log "INFO" "Installing packages..."

    if [[ "${DRY_RUN}" == true ]]; then
        log_dry "apt-get update"
        for pkg in "${PACKAGES[@]}"; do
            log_dry "apt-get install -y ${pkg}"
        done
        log_dry "Install Chromium (chromium-browser or chromium)"
        return 0
    fi

    export DEBIAN_FRONTEND=noninteractive

    log "INFO" "Updating package lists..."
    if apt-get update -y 2>&1 | tee -a "${LOG_FILE}"; then
        log "INFO" "Package lists updated"
    else
        log "WARN" "apt-get update encountered issues"
    fi

    # Install main packages
    for pkg in "${PACKAGES[@]}"; do
        log_verbose "Installing ${pkg}..."
        if apt-get install -y "${pkg}" 2>&1 | tee -a "${LOG_FILE}"; then
            PACKAGES_INSTALLED+=("${pkg}")
            log_verbose "Installed ${pkg}"
        else
            log "WARN" "Failed to install ${pkg}"
            WARNINGS+=("Package installation failed: ${pkg}")
        fi
    done

    # Handle Chromium (package name varies)
    log "INFO" "Installing Chromium..."
    if apt-cache show chromium-browser &>/dev/null; then
        if apt-get install -y chromium-browser 2>&1 | tee -a "${LOG_FILE}"; then
            PACKAGES_INSTALLED+=("chromium-browser")
            log "INFO" "Installed chromium-browser"
        else
            log "WARN" "Failed to install chromium-browser"
        fi
    elif apt-cache show chromium &>/dev/null; then
        if apt-get install -y chromium 2>&1 | tee -a "${LOG_FILE}"; then
            PACKAGES_INSTALLED+=("chromium")
            log "INFO" "Installed chromium"
        else
            log "WARN" "Failed to install chromium"
        fi
    else
        log "WARN" "Chromium package not found in repositories"
        WARNINGS+=("Chromium not available in apt repositories")
    fi

    log "INFO" "Package installation complete"
}

verify_packages() {
    log "INFO" "Verifying installed packages..."

    if [[ "${DRY_RUN}" == true ]]; then
        log_dry "Verify commands exist for installed packages"
        return 0
    fi

    local missing=()

    for pair in "${PACKAGE_COMMANDS[@]}"; do
        local pkg="${pair%%:*}"
        local cmd="${pair##*:}"
        if command -v "${cmd}" &>/dev/null; then
            log_verbose "Verified: ${cmd}"
        else
            log "WARN" "Command not found after install: ${cmd} (package: ${pkg})"
            missing+=("${pkg}")
        fi
    done

    # Check Chromium
    if command -v chromium-browser &>/dev/null; then
        log_verbose "Verified: chromium-browser"
    elif command -v chromium &>/dev/null; then
        log_verbose "Verified: chromium"
    else
        log "WARN" "Chromium command not found"
        missing+=("chromium")
    fi

    # Note about RustDesk
    if command -v rustdesk &>/dev/null; then
        log "INFO" "RustDesk is already installed"
    else
        log "INFO" "RustDesk not installed (manual .deb install required)"
        log "INFO" "Download from: https://rustdesk.com/download"
    fi

    if [[ ${#missing[@]} -gt 0 ]]; then
        log "WARN" "Some packages may not have installed correctly: ${missing[*]}"
        WARNINGS+=("Missing commands after install: ${missing[*]}")
    else
        log "INFO" "All package commands verified"
    fi
}

#-------------------------------------------------------------------------------
# Payload Deployment
#-------------------------------------------------------------------------------
deploy_payload() {
    log "INFO" "Deploying payload to ${PAYLOAD_DEST}..."

    if [[ "${DRY_RUN}" == true ]]; then
        log_dry "mkdir -p ${PAYLOAD_DEST}"
        for dir in "${PAYLOAD_DIRS[@]}"; do
            log_dry "cp -a ${PAYLOAD_SOURCE}/${dir} ${PAYLOAD_DEST}/"
        done
        return 0
    fi

    # Create destination
    mkdir -p "${PAYLOAD_DEST}"
    log_verbose "Created ${PAYLOAD_DEST}"

    # Copy each directory
    for dir in "${PAYLOAD_DIRS[@]}"; do
        local src="${PAYLOAD_SOURCE}/${dir}"
        if [[ -d "${src}" ]]; then
            log_verbose "Copying ${dir}..."
            if cp -a "${src}" "${PAYLOAD_DEST}/"; then
                FILES_DEPLOYED+=("${dir}/")
                log_verbose "Copied ${dir}"
            else
                log "WARN" "Failed to copy ${dir}"
                WARNINGS+=("Failed to copy: ${dir}")
            fi
        else
            log "WARN" "Source directory not found: ${src}"
            WARNINGS+=("Missing source directory: ${dir}")
        fi
    done

    # Set permissions
    chown -R root:root "${PAYLOAD_DEST}" 2>/dev/null || true
    find "${PAYLOAD_DEST}" -type d -exec chmod 755 {} \; 2>/dev/null || true
    find "${PAYLOAD_DEST}" -type f -exec chmod 644 {} \; 2>/dev/null || true

    log "INFO" "Payload deployment complete"
}

#-------------------------------------------------------------------------------
# Firstboot Service Setup
#-------------------------------------------------------------------------------
setup_firstboot() {
    log "INFO" "Setting up firstboot service..."

    if [[ "${DRY_RUN}" == true ]]; then
        log_dry "install -m 0755 ${FIRSTBOOT_SCRIPT_SRC} ${FIRSTBOOT_SCRIPT_DST}"
        log_dry "install -m 0644 ${FIRSTBOOT_SERVICE_SRC} ${FIRSTBOOT_SERVICE_DST}"
        log_dry "systemctl daemon-reload"
        log_dry "systemctl enable wadsworth-firstboot.service"
        return 0
    fi

    # Install firstboot script
    if [[ -f "${FIRSTBOOT_SCRIPT_SRC}" ]]; then
        install -m 0755 "${FIRSTBOOT_SCRIPT_SRC}" "${FIRSTBOOT_SCRIPT_DST}"
        FILES_DEPLOYED+=("${FIRSTBOOT_SCRIPT_DST}")
        log "INFO" "Installed firstboot script: ${FIRSTBOOT_SCRIPT_DST}"
    else
        log "ERROR" "Firstboot script not found: ${FIRSTBOOT_SCRIPT_SRC}"
        WARNINGS+=("Missing firstboot script")
        return 1
    fi

    # Install systemd service
    if [[ -f "${FIRSTBOOT_SERVICE_SRC}" ]]; then
        install -m 0644 "${FIRSTBOOT_SERVICE_SRC}" "${FIRSTBOOT_SERVICE_DST}"
        FILES_DEPLOYED+=("${FIRSTBOOT_SERVICE_DST}")
        log "INFO" "Installed systemd service: ${FIRSTBOOT_SERVICE_DST}"
    else
        log "ERROR" "Firstboot service file not found: ${FIRSTBOOT_SERVICE_SRC}"
        WARNINGS+=("Missing firstboot service file")
        return 1
    fi

    # Reload systemd and enable service
    log_verbose "Reloading systemd daemon..."
    if systemctl daemon-reload; then
        log_verbose "systemctl daemon-reload complete"
    else
        log "WARN" "systemctl daemon-reload failed"
    fi

    log_verbose "Enabling firstboot service..."
    if systemctl enable wadsworth-firstboot.service; then
        log "INFO" "Firstboot service enabled"
    else
        log "ERROR" "Failed to enable firstboot service"
        WARNINGS+=("Failed to enable firstboot service")
        return 1
    fi

    log "INFO" "Firstboot setup complete"
}

#-------------------------------------------------------------------------------
# Summary
#-------------------------------------------------------------------------------
print_summary() {
    echo ""
    echo "=========================================="
    echo "  Wadsworth Golden Image Installer"
    echo "  Installation Summary"
    echo "=========================================="
    echo ""

    if [[ "${DRY_RUN}" == true ]]; then
        echo "MODE: Dry-run (no changes made)"
        echo ""
    fi

    echo "PACKAGES INSTALLED:"
    if [[ ${#PACKAGES_INSTALLED[@]} -gt 0 ]]; then
        for pkg in "${PACKAGES_INSTALLED[@]}"; do
            echo "  - ${pkg}"
        done
    else
        if [[ "${DRY_RUN}" == true ]]; then
            for pkg in "${PACKAGES[@]}"; do
                echo "  - ${pkg} (would install)"
            done
            echo "  - chromium/chromium-browser (would install)"
        else
            echo "  (none)"
        fi
    fi
    echo ""

    echo "FILES DEPLOYED:"
    if [[ ${#FILES_DEPLOYED[@]} -gt 0 ]]; then
        for file in "${FILES_DEPLOYED[@]}"; do
            echo "  - ${file}"
        done
    else
        if [[ "${DRY_RUN}" == true ]]; then
            for dir in "${PAYLOAD_DIRS[@]}"; do
                echo "  - ${PAYLOAD_DEST}/${dir}/ (would copy)"
            done
            echo "  - ${FIRSTBOOT_SCRIPT_DST} (would install)"
            echo "  - ${FIRSTBOOT_SERVICE_DST} (would install)"
        else
            echo "  (none)"
        fi
    fi
    echo ""

    if [[ ${#WARNINGS[@]} -gt 0 ]]; then
        echo "WARNINGS:"
        for warning in "${WARNINGS[@]}"; do
            echo "  ! ${warning}"
        done
        echo ""
    fi

    echo "NEXT STEPS:"
    echo "  1. Install RustDesk manually:"
    echo "     - Download from https://rustdesk.com/download"
    echo "     - sudo dpkg -i rustdesk-*.deb"
    echo "     - sudo apt-get install -f  (if needed)"
    echo ""
    echo "  2. Verify installation:"
    echo "     - systemctl is-enabled wadsworth-firstboot.service"
    echo "     - ls -la ${PAYLOAD_DEST}/"
    echo ""
    echo "  3. Do NOT reboot this machine"
    echo "     (firstboot runs on deployed units only)"
    echo ""
    echo "  4. Capture golden image with Clonezilla"
    echo ""
    echo "  5. Deploy to target machines"
    echo "     - Target boots -> firstboot runs -> ready for customer"
    echo ""
    echo "=========================================="

    if [[ "${DRY_RUN}" == true ]]; then
        echo ""
        echo "This was a dry-run. No changes were made."
        echo "Run without --dry-run to apply changes."
    fi

    log "INFO" "Installation complete"
}

#-------------------------------------------------------------------------------
# Main Execution
#-------------------------------------------------------------------------------
main() {
    # Parse command-line arguments
    parse_args "$@"

    # Show dry-run notice
    if [[ "${DRY_RUN}" == true ]]; then
        echo "============================================"
        echo "  DRY-RUN MODE - No changes will be made"
        echo "============================================"
        echo ""
    fi

    # Pre-flight checks
    check_root
    init_logging
    check_os

    # Installation steps
    install_packages
    verify_packages
    deploy_payload
    setup_firstboot

    # Summary
    print_summary

    exit 0
}

# Run main function
main "$@"
