#!/usr/bin/env bash
#===============================================================================
# Wadsworth AI PCs — First Boot Personalization Script (Production v1.0)
#
# This script runs ONCE after a golden image is deployed to personalize each
# machine. It is triggered by wadsworth-firstboot.service on first boot.
#
# IMPORTANT:
# - Do NOT configure unattended remote access
# - All operations should be idempotent and fail gracefully
# - Comprehensive logging for troubleshooting
#===============================================================================

set -euo pipefail

#-------------------------------------------------------------------------------
# Configuration
#-------------------------------------------------------------------------------
readonly LOG_DIR="/var/log/wadsworth"
readonly LOG_FILE="${LOG_DIR}/firstboot.log"
readonly MARKER_FILE="/var/lib/wadsworth/firstboot.done"
readonly MARKER_DIR="/var/lib/wadsworth"
readonly PAYLOAD_SOURCE="/opt/wadsworth_payload"
readonly PAYLOAD_DEST="/opt/wadsworth"
readonly WALLPAPER_PATH="${PAYLOAD_DEST}/assets/wadsworth_wallpaper.png"

#-------------------------------------------------------------------------------
# Logging Function
#-------------------------------------------------------------------------------
log() {
    local level="${1:-INFO}"
    shift
    local message="$*"
    local timestamp
    timestamp="$(date -Iseconds)"
    echo "[${timestamp}] [${level}] ${message}" | tee -a "${LOG_FILE}"
}

#-------------------------------------------------------------------------------
# Initialize Logging
#-------------------------------------------------------------------------------
init_logging() {
    mkdir -p "${LOG_DIR}"
    touch "${LOG_FILE}"
    chmod 644 "${LOG_FILE}"
    log "INFO" "=========================================="
    log "INFO" "Wadsworth First Boot Script Starting"
    log "INFO" "=========================================="
}

#-------------------------------------------------------------------------------
# Run-Once Guard
#-------------------------------------------------------------------------------
check_marker() {
    if [[ -f "${MARKER_FILE}" ]]; then
        log "INFO" "Marker file exists: ${MARKER_FILE}"
        log "INFO" "First boot already completed. Exiting cleanly."
        exit 0
    fi
    log "INFO" "No marker file found. Proceeding with first boot setup."
}

#-------------------------------------------------------------------------------
# Machine-ID Regeneration
#-------------------------------------------------------------------------------
regenerate_machine_id() {
    log "INFO" "Regenerating machine-id..."

    if [[ -f /etc/machine-id ]]; then
        rm -f /etc/machine-id
        log "INFO" "Removed /etc/machine-id"
    fi

    if [[ -f /var/lib/dbus/machine-id ]]; then
        rm -f /var/lib/dbus/machine-id
        log "INFO" "Removed /var/lib/dbus/machine-id"
    fi

    if systemd-machine-id-setup; then
        log "INFO" "New machine-id generated: $(cat /etc/machine-id)"
    else
        log "ERROR" "Failed to generate machine-id"
        return 1
    fi
}

#-------------------------------------------------------------------------------
# SSH Host Key Regeneration
#-------------------------------------------------------------------------------
regenerate_ssh_keys() {
    log "INFO" "Checking SSH host key regeneration..."

    if ! command -v ssh-keygen &>/dev/null; then
        log "INFO" "openssh-server not installed. Skipping SSH key regeneration."
        return 0
    fi

    if [[ ! -d /etc/ssh ]]; then
        log "INFO" "/etc/ssh does not exist. Skipping SSH key regeneration."
        return 0
    fi

    log "INFO" "Removing existing SSH host keys..."
    rm -f /etc/ssh/ssh_host_* 2>/dev/null || true

    log "INFO" "Regenerating SSH host keys..."
    if dpkg-reconfigure openssh-server 2>/dev/null; then
        log "INFO" "SSH host keys regenerated successfully"
    else
        log "WARN" "dpkg-reconfigure openssh-server failed (may be expected)"
    fi
}

#-------------------------------------------------------------------------------
# Hostname Generation
#-------------------------------------------------------------------------------
sanitize_hostname_part() {
    # Lowercase, alphanumeric + hyphens only, trim leading/trailing hyphens
    echo "$1" | tr '[:upper:]' '[:lower:]' | tr -cs 'a-z0-9-' '-' | sed 's/^-//;s/-$//'
}

generate_hostname() {
    log "INFO" "Generating hostname..."

    local product_name serial_number model_part suffix new_hostname

    # Read product name from DMI
    if [[ -f /sys/devices/virtual/dmi/id/product_name ]]; then
        product_name="$(cat /sys/devices/virtual/dmi/id/product_name 2>/dev/null || echo "")"
    else
        product_name=""
    fi

    # Read serial number from DMI
    if [[ -f /sys/devices/virtual/dmi/id/product_serial ]]; then
        serial_number="$(cat /sys/devices/virtual/dmi/id/product_serial 2>/dev/null || echo "")"
    else
        serial_number=""
    fi

    log "INFO" "Product name: ${product_name:-unknown}"
    log "INFO" "Serial number: ${serial_number:-unavailable}"

    # Sanitize model name (max 15 chars for model portion)
    if [[ -n "${product_name}" ]]; then
        model_part="$(sanitize_hostname_part "${product_name}" | cut -c1-15)"
    else
        model_part="pc"
    fi

    # Generate suffix from serial (last 4 chars) or random
    if [[ -n "${serial_number}" && "${serial_number}" != "To Be Filled By O.E.M." && "${serial_number}" != "None" ]]; then
        # Extract last 4 alphanumeric characters
        suffix="$(echo "${serial_number}" | tr -dc 'A-Za-z0-9' | tail -c 4)"
        suffix="${suffix,,}"  # Lowercase
    else
        log "INFO" "Serial unavailable, using random suffix"
        suffix="$(head -c 100 /dev/urandom | tr -dc 'a-z0-9' | head -c 4)"
    fi

    # Ensure suffix is not empty
    if [[ -z "${suffix}" ]]; then
        suffix="$(head -c 100 /dev/urandom | tr -dc 'a-z0-9' | head -c 4)"
    fi

    new_hostname="wadsworth-${model_part}-${suffix}"

    log "INFO" "Setting hostname to: ${new_hostname}"

    if hostnamectl set-hostname "${new_hostname}"; then
        log "INFO" "Hostname set successfully"
    else
        log "ERROR" "Failed to set hostname"
        return 1
    fi
}

#-------------------------------------------------------------------------------
# Payload Deployment
#-------------------------------------------------------------------------------
deploy_payload() {
    log "INFO" "Deploying Wadsworth payload..."

    if [[ ! -d "${PAYLOAD_SOURCE}" ]]; then
        log "WARN" "Payload source not found: ${PAYLOAD_SOURCE}"
        log "WARN" "Skipping payload deployment (expected during production builds)"
        return 0
    fi

    # Create destination directory structure
    mkdir -p "${PAYLOAD_DEST}"

    # List of directories to copy
    local dirs=("ai_hub" "prompt_pack" "welcome_center" "docs" "assets" "desktop_shortcuts" "bin")

    for dir in "${dirs[@]}"; do
        if [[ -d "${PAYLOAD_SOURCE}/${dir}" ]]; then
            log "INFO" "Copying ${dir}..."
            cp -a "${PAYLOAD_SOURCE}/${dir}" "${PAYLOAD_DEST}/" 2>/dev/null || {
                log "WARN" "Failed to copy ${dir}"
            }
        else
            log "WARN" "Directory not found: ${PAYLOAD_SOURCE}/${dir}"
        fi
    done

    # Set proper permissions
    chown -R root:root "${PAYLOAD_DEST}" 2>/dev/null || true
    find "${PAYLOAD_DEST}" -type d -exec chmod 755 {} \; 2>/dev/null || true
    find "${PAYLOAD_DEST}" -type f -exec chmod 644 {} \; 2>/dev/null || true
    find "${PAYLOAD_DEST}" -type f -name "*.sh" -exec chmod 755 {} \; 2>/dev/null || true

    log "INFO" "Payload deployment complete"
}

#-------------------------------------------------------------------------------
# Primary User Detection
#-------------------------------------------------------------------------------
get_primary_user() {
    local username uid home

    while IFS=: read -r username _ uid _ _ home _; do
        if [[ ${uid} -ge 1000 ]] && [[ -d "${home}" ]] && [[ "${username}" != "nobody" ]]; then
            echo "${username}"
            return 0
        fi
    done < /etc/passwd

    return 1
}

detect_primary_user() {
    log "INFO" "Detecting primary user..."

    PRIMARY_USER=""
    PRIMARY_HOME=""

    if PRIMARY_USER="$(get_primary_user)"; then
        PRIMARY_HOME="$(getent passwd "${PRIMARY_USER}" | cut -d: -f6)"
        log "INFO" "Primary user detected: ${PRIMARY_USER}"
        log "INFO" "Home directory: ${PRIMARY_HOME}"
    else
        log "WARN" "No primary user found (UID >= 1000 with valid home)"
        return 1
    fi
}

#-------------------------------------------------------------------------------
# Desktop Shortcuts Deployment
#-------------------------------------------------------------------------------
deploy_desktop_shortcuts() {
    log "INFO" "Deploying desktop shortcuts..."

    if [[ -z "${PRIMARY_USER:-}" ]] || [[ -z "${PRIMARY_HOME:-}" ]]; then
        log "WARN" "Primary user not set. Skipping desktop shortcuts."
        return 0
    fi

    if [[ ! -d "${PRIMARY_HOME}" ]]; then
        log "WARN" "Primary user home does not exist: ${PRIMARY_HOME}"
        return 0
    fi

    local desktop_dir="${PRIMARY_HOME}/Desktop"
    local ai_dir="${desktop_dir}/AI"
    local shortcuts_source="${PAYLOAD_DEST}/desktop_shortcuts"
    local primary_uid primary_gid

    primary_uid="$(id -u "${PRIMARY_USER}")"
    primary_gid="$(id -g "${PRIMARY_USER}")"

    # Create Desktop directory if it doesn't exist
    if [[ ! -d "${desktop_dir}" ]]; then
        mkdir -p "${desktop_dir}"
        chown "${PRIMARY_USER}:${PRIMARY_USER}" "${desktop_dir}"
        log "INFO" "Created Desktop directory"
    fi

    # Create AI subfolder
    mkdir -p "${ai_dir}"
    chown "${PRIMARY_USER}:${PRIMARY_USER}" "${ai_dir}"
    log "INFO" "Created AI folder on Desktop"

    if [[ ! -d "${shortcuts_source}" ]]; then
        log "WARN" "Desktop shortcuts source not found: ${shortcuts_source}"
        return 0
    fi

    # AI-related shortcuts go in ~/Desktop/AI/
    local ai_shortcuts=("chatgpt.desktop" "claude.desktop" "gemini.desktop" "ai-hub.desktop")

    for shortcut in "${ai_shortcuts[@]}"; do
        if [[ -f "${shortcuts_source}/${shortcut}" ]]; then
            cp "${shortcuts_source}/${shortcut}" "${ai_dir}/"
            chown "${PRIMARY_USER}:${PRIMARY_USER}" "${ai_dir}/${shortcut}"
            chmod +x "${ai_dir}/${shortcut}"
            log "INFO" "Deployed ${shortcut} to AI folder"

            # Trust the desktop file using gio
            if command -v gio &>/dev/null; then
                sudo -u "${PRIMARY_USER}" \
                    DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/${primary_uid}/bus" \
                    gio set "${ai_dir}/${shortcut}" metadata::trusted true 2>/dev/null || {
                    # Fallback: try dbus-launch method
                    sudo -u "${PRIMARY_USER}" dbus-launch gio set "${ai_dir}/${shortcut}" metadata::trusted true 2>/dev/null || true
                }
            fi
        fi
    done

    # Welcome Center shortcut goes directly on Desktop
    if [[ -f "${shortcuts_source}/welcome-center.desktop" ]]; then
        cp "${shortcuts_source}/welcome-center.desktop" "${desktop_dir}/"
        chown "${PRIMARY_USER}:${PRIMARY_USER}" "${desktop_dir}/welcome-center.desktop"
        chmod +x "${desktop_dir}/welcome-center.desktop"
        log "INFO" "Deployed Welcome Center to Desktop"

        if command -v gio &>/dev/null; then
            sudo -u "${PRIMARY_USER}" \
                DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/${primary_uid}/bus" \
                gio set "${desktop_dir}/welcome-center.desktop" metadata::trusted true 2>/dev/null || {
                sudo -u "${PRIMARY_USER}" dbus-launch gio set "${desktop_dir}/welcome-center.desktop" metadata::trusted true 2>/dev/null || true
            }
        fi
    fi

    log "INFO" "Desktop shortcuts deployment complete"
}

#-------------------------------------------------------------------------------
# Wallpaper Branding (Cinnamon)
#-------------------------------------------------------------------------------
set_wallpaper() {
    log "INFO" "Setting desktop wallpaper..."

    if [[ -z "${PRIMARY_USER:-}" ]]; then
        log "WARN" "Primary user not set. Skipping wallpaper configuration."
        return 0
    fi

    if [[ ! -f "${WALLPAPER_PATH}" ]]; then
        log "WARN" "Wallpaper not found: ${WALLPAPER_PATH}"
        log "WARN" "Skipping wallpaper configuration"
        return 0
    fi

    local primary_uid
    primary_uid="$(id -u "${PRIMARY_USER}")"

    log "INFO" "Applying wallpaper: ${WALLPAPER_PATH}"

    # Set wallpaper using gsettings as the primary user
    if sudo -u "${PRIMARY_USER}" \
        DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/${primary_uid}/bus" \
        gsettings set org.cinnamon.desktop.background picture-uri "file://${WALLPAPER_PATH}" 2>/dev/null; then
        log "INFO" "Wallpaper URI set successfully"
    else
        # Fallback: try dbus-launch method
        if sudo -u "${PRIMARY_USER}" dbus-launch \
            gsettings set org.cinnamon.desktop.background picture-uri "file://${WALLPAPER_PATH}" 2>/dev/null; then
            log "INFO" "Wallpaper URI set successfully (via dbus-launch)"
        else
            log "WARN" "Failed to set wallpaper (user may not have active session)"
        fi
    fi

    # Set picture options to zoom
    sudo -u "${PRIMARY_USER}" \
        DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/${primary_uid}/bus" \
        gsettings set org.cinnamon.desktop.background picture-options "zoom" 2>/dev/null || true
}

#-------------------------------------------------------------------------------
# System Updates
#-------------------------------------------------------------------------------
run_system_updates() {
    log "INFO" "Running system updates (non-interactive)..."

    export DEBIAN_FRONTEND=noninteractive

    log "INFO" "Running apt-get update..."
    if apt-get update -y 2>&1 | tee -a "${LOG_FILE}"; then
        log "INFO" "Package lists updated successfully"
    else
        log "WARN" "apt-get update encountered issues"
    fi

    log "INFO" "Running apt-get upgrade..."
    if apt-get upgrade -y 2>&1 | tee -a "${LOG_FILE}"; then
        log "INFO" "System packages upgraded successfully"
    else
        log "WARN" "apt-get upgrade encountered issues"
    fi
}

#-------------------------------------------------------------------------------
# Timeshift Configuration and Snapshot
#-------------------------------------------------------------------------------
configure_timeshift() {
    log "INFO" "Configuring Timeshift..."

    if ! command -v timeshift &>/dev/null; then
        log "WARN" "Timeshift not installed. Skipping snapshot creation."
        return 0
    fi

    local timeshift_config="/etc/timeshift/timeshift.json"

    # Create config directory
    mkdir -p /etc/timeshift

    # Only write config if it doesn't exist or has do_first_run set to true
    if [[ ! -f "${timeshift_config}" ]]; then
        log "INFO" "Creating Timeshift configuration for RSYNC mode..."

        cat > "${timeshift_config}" << 'EOF'
{
  "backup_device_uuid" : "",
  "parent_device_uuid" : "",
  "do_first_run" : "false",
  "btrfs_mode" : "false",
  "include_btrfs_home_for_backup" : "false",
  "include_btrfs_home_for_restore" : "false",
  "stop_cron_emails" : "true",
  "schedule_monthly" : "false",
  "schedule_weekly" : "false",
  "schedule_daily" : "false",
  "schedule_hourly" : "false",
  "schedule_boot" : "false",
  "count_monthly" : "2",
  "count_weekly" : "3",
  "count_daily" : "5",
  "count_hourly" : "6",
  "count_boot" : "5",
  "snapshot_size" : "0",
  "snapshot_count" : "0",
  "exclude" : [],
  "exclude-apps" : []
}
EOF
        log "INFO" "Timeshift configuration written"
    else
        log "INFO" "Timeshift configuration already exists"
    fi

    # Create Factory Fresh snapshot
    log "INFO" "Creating 'Factory Fresh' Timeshift snapshot..."
    if timeshift --create --comments "Factory Fresh" --tags D 2>&1 | tee -a "${LOG_FILE}"; then
        log "INFO" "Factory Fresh snapshot created successfully"
    else
        log "WARN" "Failed to create Timeshift snapshot (Timeshift may need manual configuration)"
    fi
}

#-------------------------------------------------------------------------------
# RustDesk Verification
#-------------------------------------------------------------------------------
verify_rustdesk() {
    log "INFO" "Verifying RustDesk installation..."

    if command -v rustdesk &>/dev/null; then
        log "INFO" "RustDesk is installed"
        local version
        version="$(rustdesk --version 2>/dev/null || echo 'unknown')"
        log "INFO" "RustDesk version: ${version}"
    elif [[ -f /usr/bin/rustdesk ]] || [[ -f /opt/rustdesk/rustdesk ]]; then
        log "INFO" "RustDesk binary found"
    else
        log "WARN" "RustDesk is NOT installed"
        log "WARN" "Remote support will not be available until RustDesk is installed"
    fi

    # Explicitly NOT configuring any automatic/unattended access
    log "INFO" "Note: RustDesk unattended access is NOT configured (by design)"
}

#-------------------------------------------------------------------------------
# Completion
#-------------------------------------------------------------------------------
mark_complete() {
    log "INFO" "Marking first boot as complete..."

    mkdir -p "${MARKER_DIR}"
    date -Iseconds > "${MARKER_FILE}"
    chmod 644 "${MARKER_FILE}"

    log "INFO" "Marker file written: ${MARKER_FILE}"
    log "INFO" "=========================================="
    log "INFO" "Wadsworth First Boot Complete!"
    log "INFO" "=========================================="
}

#-------------------------------------------------------------------------------
# Main Execution
#-------------------------------------------------------------------------------
main() {
    # Initialize
    init_logging
    check_marker

    # Machine identity
    regenerate_machine_id
    regenerate_ssh_keys
    generate_hostname

    # Payload and user setup
    deploy_payload
    detect_primary_user || true  # Continue even if no user found
    deploy_desktop_shortcuts
    set_wallpaper

    # System maintenance
    run_system_updates
    configure_timeshift

    # Verification
    verify_rustdesk

    # Complete
    mark_complete

    exit 0
}

# Run main function
main "$@"
